<?php
/**
 * SEO
 *
 * @package Atlas Core
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/*-----------------------------------------------------------------------------------*/
# site description
/*-----------------------------------------------------------------------------------*/
if ( ! function_exists( 'th90_site_description' ) ) {
    function th90_site_description() {

		if ( ! is_front_page() ) {
			return false;
		}

		$content = th90_opt( 'site_desc' );
		if ( ! empty ( $content ) ) {
			echo '<meta name="description" content="' . esc_html( $content ) . '">';
		}

		return false;
	}
}

add_action( 'wp_head', 'th90_site_description', 1 );


/*-----------------------------------------------------------------------------------*/
# Open Graph Meta for posts
/*-----------------------------------------------------------------------------------*/
if( ! function_exists( 'th90_opengraph' ) ){

	add_action( 'wp_head', 'th90_opengraph', 5 );
	function th90_opengraph(){

		# Check if single and og is active and there is no OG plugin is active ----------
		if( th90_is_opengraph_active() || ! is_singular( 'post' ) || ! th90_opt( 'post_og_cards' )){
			return;
		}
		$og_title       = the_title_attribute( 'echo=0' ) . ' - ' . get_bloginfo('name') ;
		$og_description = th90_get_excerpt();
		$og_type        = 'article';
        $facebook_app_id = th90_opt( 'facebook_app_id' );

		if( is_home() || is_front_page() ){
			$og_title       = get_bloginfo( 'name' );
			$og_description = get_bloginfo( 'description' );
			$og_type        = 'website';
		}
        ?>
        <meta property="og:title" content="<?php echo esc_attr( $og_title ); ?>" />
        <meta property="og:type" content="<?php echo esc_attr( $og_type ); ?>" />
        <meta property="og:description" content="<?php echo esc_attr( wp_html_excerpt( $og_description, 100 ) ); ?>" />
        <meta property="og:url" content="<?php echo esc_url( get_permalink() ); ?>" />
        <meta property="og:site_name" content="<?php echo esc_attr( get_bloginfo( 'name' ) ); ?>" />


        <?php if ( has_post_thumbnail() ) : ?>
			<meta property="og:image" content="<?php echo esc_url( th90_post_thumbnail_url() ); ?>"/>
		<?php else : ?>
            <?php
			$logo = th90_opt_arr( 'facebook_default_img', 'url' );
			if ( empty( $logo ) ) {
				$logo = th90_opt_arr( 'logo', 'url' );
			}
			if ( ! empty( $logo ) ): ?>
				<meta property="og:image" content="<?php echo esc_url( $logo ); ?>"/>
			<?php endif; ?>
		<?php endif; ?>

		<?php if ( ! empty( $facebook_app_id ) ) : ?>
			<meta property="fb:facebook_app_id" content="<?php echo esc_attr( $facebook_app_id ); ?>"/>
		<?php endif; ?>

        <?php
	}

}

/*-----------------------------------------------------------------------------------*/
# Add the opengraph namespace to the <html> tag
/*-----------------------------------------------------------------------------------*/
if( ! function_exists( 'th90_add_opengraph_namespace' ) ){

	add_filter( 'language_attributes', 'th90_add_opengraph_namespace' );
	function th90_add_opengraph_namespace( $input ){

		# Check if single and og is active and there is no OG plugin is active ----------
		if( is_admin() || th90_is_opengraph_active() || ! is_singular() || ! th90_opt( 'post_og_cards' )){
			return $input;
		}

		return $input.' prefix="og: https://ogp.me/ns#"';
	}

}

/*-----------------------------------------------------------------------------------*/
# Check if a an open graph plugin active
/*-----------------------------------------------------------------------------------*/
if( ! function_exists( 'th90_is_opengraph_active' ) ){

	function th90_is_opengraph_active(){

		# Yoast SEO ----------
		if( class_exists( 'WPSEO_Frontend' ) ){
			$yoast = get_option( 'wpseo_social' );
			if( ! empty( $yoast['opengraph'] )){
				return true;
			}
		}

		# Jetpack ----------
		if ( TH90_JETPACK_IS_ACTIVE && ( in_array( 'publicize', Jetpack::get_active_modules() ) || in_array( 'sharedaddy', Jetpack::get_active_modules() ) ) ){
			return true;
		}

		# Else ----------
		return false;
	}

}

/*-----------------------------------------------------------------------------------*/
# Rich Snippets
/*-----------------------------------------------------------------------------------*/
if( ! function_exists( 'th90_article_schemas' )){

	function th90_article_schemas(){

		if( ! th90_opt( 'post_rich_snippet' ) ) {
			return false;
		}

		$post        = get_post();
		$post_id     = $post->ID;
		$has_review  = th90_field_single( 'review_show' ) ? true : false;
		$schema_type = ( $has_review ) ? 'review' : th90_opt( 'post_schema_type' );


		# Is the current is a normal page and without review return ----------
		if( ! $has_review && is_page() ){
			return;
		}


		# Site Logo ----------
		$site_logo = th90_opt_arr( 'logo', 'url' );


		# Tags and Categories ----------
		$tags = th90_get_plain_terms( $post_id, 'post_tag' );
		$cats = th90_get_plain_terms( $post_id, 'category' );


		# Post data ----------
		$description    = th90_get_excerpt();
		$puplished_date = ( get_the_time( 'c' ) ) ? get_the_time( 'c' ) : get_the_modified_date( 'c' );
		$modified_date  = ( get_the_modified_date( 'c' ) ) ? get_the_modified_date( 'c' ) : $puplished_date;


		# The Scemas Array ----------
		$schema = array(
			'@context'       => th90_ssl() . 'schema.org',
			'@type'          => $schema_type,
			'dateCreated'    => $puplished_date,
			'datePublished'  => $puplished_date,
			'dateModified'   => $modified_date,
			'headline'       => strip_tags( get_the_title() ),
			'name'           => strip_tags( get_the_title() ),
			'keywords'       => $tags,
			'url'            => get_permalink(),
			'description'    => $description,
			'copyrightYear'  => get_the_time( 'Y' ),
			'publisher'      => array(
				'@id'   => '#Publisher',
				'@type' => 'Organization',
				'name'  => get_bloginfo(),
				'logo'  => array(
						'@type'  => 'ImageObject',
						'url'    => $site_logo,
				)
			),
			'sourceOrganization' => array(
				'@id' => '#Publisher'
			),
			'copyrightHolder'    => array(
				'@id' => '#Publisher'
			),
			'mainEntityOfPage' => array(
				'@type'      => 'WebPage',
				'@id'        => get_permalink(),
			),
			'author' => array(
				'@type' => 'Person',
				'name'  => get_the_author(),
				'url'   => get_author_posts_url( get_the_author_meta( 'ID' ) ),
			),
		);


		# Breadcrumbs ----------
		if( th90_opt( 'breadcrumb' ) ){
			$schema['mainEntityOfPage']['breadcrumb'] = array(
				'@id' => '#Breadcrumb'
			);
		}


		# Review ----------
		if( ! empty( $has_review ) ){

			# Get the summary and the total score ----------
			$total_score = get_post_meta( $post_id, 'th90_review_total_score', true );
			//$review_summary = th90_get_excerpt();


			# Convert the total score to 0-5 rating ----------
			if( ! empty( $total_score ) && $total_score > 0 ){
				$total_score = round( ( $total_score*5 )/100, 1 );
			}

			# Add the review to the schema array ----------
			$schema['itemReviewed'] = array(
				'@type' => 'Thing',
				'name'  => get_the_title(),
			);

			$schema['reviewRating'] = array(
				'@type'       => 'Rating',
				'worstRating' => 1,
				'bestRating'  => 5,
				'ratingValue' => $total_score,
				//'description' => $review_summary,
			);

			$schema['reviewBody'] = $description;

		}

		# It is not review so add articleSection ----------
		else{
			$schema['articleSection'] = $cats;
		}


		# Post image ----------
		$image_id   = get_post_thumbnail_id();
		$image_data = wp_get_attachment_image_src( $image_id, 'full' );

		if( ! empty( $image_data ) ){
			$schema['image'] = array(
				'@type'  => 'ImageObject',
				'url'    => $image_data[0],
				'width'  => ( $image_data[1] > 696 ) ? $image_data[1] : 696,
				'height' => $image_data[2],
			);

			if( ! empty( $has_review ) ){
				$schema['itemReviewed']['image'] = $image_data[0];
			}
		}


		# Print the schema ----------
		echo '<script type="application/ld+json">'. json_encode( $schema ) .'</script>';
	}

	add_action( 'th90_article_elements_below', 'th90_article_schemas', 40 );

}

/*-----------------------------------------------------------------------------------*/
# get terms as plain text seprated with commas
/*-----------------------------------------------------------------------------------*/
function th90_get_plain_terms( $post_id, $term ){

	$post_terms = get_the_terms( $post_id, $term );

	$terms = array();

	if( ! empty( $post_terms ) && is_array( $post_terms ) ){
		foreach ( $post_terms as $term ) {
			$terms[] = $term->name;
		}

		$terms = implode( ',', $terms );
	}

	return $terms;
}

/*-----------------------------------------------------------------------------------*/
# link search box schema
/*-----------------------------------------------------------------------------------*/
if ( ! function_exists( 'th90_schema_website' ) ) {
    function th90_schema_website() {

		$schema = th90_opt( 'site_markup' );
		if ( empty( $schema ) ) {
			return false;
		}

		$home_url = home_url( '/' );
		$json_ld  = array(
			'@context'        => th90_ssl() . 'schema.org',
			'@type'           => 'WebSite',
			'@id'             => $home_url . '#website',
			'url'             => $home_url,
			'name'            => get_bloginfo( 'name' ),
			'potentialAction' => array(
				'@type'       => 'SearchAction',
				'target'      => $home_url . '?s={search_term_string}',
				'query-input' => 'required name=search_term_string',
			),
		);

		echo '<script type="application/ld+json">';
		if ( version_compare( PHP_VERSION, '5.4', '>=' ) ) {
			echo wp_json_encode( $json_ld, JSON_UNESCAPED_SLASHES );
		} else {
			echo wp_json_encode( $json_ld );
		}
		echo '</script>', "\n";

		return false;
	}
}
add_action( 'wp_head', 'th90_schema_website', 10 );
