<?php
/**
 * Images Lazyload Functions
 *
 * @package Atlas
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * -----------------------------------------------------------------------------
 * Lazyload Image
 * -----------------------------------------------------------------------------
 */
if ( ! function_exists( 'th90_add_lazyload' ) ) {
	function th90_add_lazyload( $img_html, $img_id ) {

		if ( ! th90_lazyload_check() || false !== strpos( $img_html, 'data-lazy-src' ) || false !== strpos( $img_html, 'data-src' ) || false !== strpos( $img_html, 'th90-no-lazy' ) ) {
			return $img_html;
		} else {
			$default_placeholder = 'data:image/gif;base64,R0lGODlhAQABAAAAACH5BAEKAAEALAAAAAABAAEAAAICTAEAOw==';
			if ( $img_id && ! th90_opt( 'disable_lazyload_img_placeholder' ) ) {
				$img = wp_get_attachment_image_src( $img_id, 'placeholder-img' );

				if ( true == $img[3] ) {
			        $placeholder_url = apply_filters( 'th90_lazy_placeholder', $img[0] );
			    } else {
			        $placeholder_url = apply_filters( 'th90_lazy_placeholder', $default_placeholder );
			    }
			} else {
				$placeholder_url = apply_filters( 'th90_lazy_placeholder', $default_placeholder );
			}

			$img_html = preg_replace( '/<img(.*?)src=/is', '<img$1src="' . esc_attr( $placeholder_url ) . '" data-src=', $img_html );
			$img_html = str_replace( 'srcset', 'data-srcset', $img_html );
			$img_html = preg_replace( '/class=(["\'])(.*?)["\']/is', 'class=$1lazyload $2$1', $img_html );
			$img_html = preg_replace( '/(sizes)="[^"]*"/', 'data-sizes="auto" data-expand="0"', $img_html );

			return $img_html;
		}
	}
}

if ( ! function_exists( 'th90_core_the_add_lazyload' ) ) {
	function th90_core_the_add_lazyload( $img_html, $img_id = null ) {

		echo th90_add_lazyload( $img_html, $img_id );
	}
}


/**
 * -----------------------------------------------------------------------------
 * Featured Image Lazyload
 * -----------------------------------------------------------------------------
 */
if ( ! function_exists( 'th90_add_lazyload_featured' ) ) {
	function th90_add_lazyload_featured( $img_html, $post_id, $post_thumbnail_id, $size, $attr ) {

		return th90_add_lazyload( $img_html, $post_thumbnail_id );
	}
}
add_filter( 'post_thumbnail_html', 'th90_add_lazyload_featured', 10, 5 );

/**
 * -----------------------------------------------------------------------------
 * Single Content Image & iframe Lazyload
 * -----------------------------------------------------------------------------
 */
if ( ! function_exists( 'th90_add_lazyload_content' ) ) {
	function th90_add_lazyload_content( $content ) {

		if ( ! th90_lazyload_check() || false !== strpos( $content, 'data-lazy-src' ) || false !== strpos( $content, 'data-src' ) || false !== strpos( $content, 'th90-no-lazy' ) ) {
			return $content;
		} else {
			if ( is_single() ) {
				$matches_images = array();
				$matches_iframes = array();
				$search = array();
				$replace = array();

				preg_match_all( '/<img[\s\r\n]+.*?>/is', $content, $matches_images );

				foreach ( $matches_images[0] as $imgHTML ) {
					// don't do the replacement if the image is a data-uri
					if ( ! preg_match( "/src=['\"]data:image/is", $imgHTML ) ) {
						$placeholder_url = 'data:image/gif;base64,R0lGODlhAQABAAAAACH5BAEKAAEALAAAAAABAAEAAAICTAEAOw==';

						// replace the src and add the data-src attribute
						$replaceHTML = preg_replace( '/<img(.*?)src=/is', '<img$1src="' . esc_attr( $placeholder_url ) . '" data-src=', $imgHTML );

						// also replace the srcset (responsive images)
						$replaceHTML = str_replace( 'srcset', 'data-srcset', $replaceHTML );

						// replace sizes to avoid w3c errors for missing srcset
						$replaceHTML = preg_replace( '/(sizes)="[^"]*"/', 'data-sizes="auto" data-expand="0"', $replaceHTML );

						// add the lazy class to the img element
						if ( preg_match( '/class=["\']/i', $replaceHTML ) ) {
							$replaceHTML = preg_replace( '/class=(["\'])(.*?)["\']/is', 'class=$1lazyload $2$1', $replaceHTML );
						} else {
							$replaceHTML = preg_replace( '/<img/is', '<img class="lazyload"', $replaceHTML );
						}

						array_push( $search, $imgHTML );
						array_push( $replace, $replaceHTML );
					}
				}

				preg_match_all( '/<div\s+class="wp-block-embed__wrapper"\s*>\s*<iframe[\s\r\n]+.*?>/is', $content, $matches_iframes );

				foreach ( $matches_iframes[0] as $iframeHTML ) {
					$replaceHTML = preg_replace( '/<iframe(.*?)src=/is', '<iframe$1class="lazyload" data-src=', $iframeHTML );
					array_push( $search, $iframeHTML );
					array_push( $replace, $replaceHTML );
				}

				$content = str_replace( $search, $replace, $content );

				return $content;
			}
			return $content;
		}
	}
}
add_filter( 'the_content', 'th90_add_lazyload_content' );


/**
 * -----------------------------------------------------------------------------
 * Gravatar Lazyload
 * -----------------------------------------------------------------------------
 */
if ( ! function_exists( 'th90_add_lazyload_avatar' ) ) {

	function th90_add_lazyload_avatar( $avatar, $user, $size, $default, $alt, $args ) {

		return th90_add_lazyload( $avatar, null );
	}
}
add_filter( 'get_avatar', 'th90_add_lazyload_avatar', 10, 6 );

/**
 * -----------------------------------------------------------------------------
 * Iframe Lazyload
 * -----------------------------------------------------------------------------
 */
if ( ! function_exists( 'th90_add_lazyload_iframe' ) ) {
	function th90_add_lazyload_iframe( $iframe_html ) {
		if ( ! th90_lazyload_check() || false !== strpos( $iframe_html, 'data-lazy-src' ) || false !== strpos( $iframe_html, 'data-src' ) || false !== strpos( $iframe_html, 'th90-no-lazy' ) ) {
			return $iframe_html;
		} else {
			$iframe_html = preg_replace( '/<iframe(.*?)src=/is', '<iframe$1class="lazyload" data-src=', $iframe_html );
			return $iframe_html;
		}
	}
}

if ( ! function_exists( 'th90_core_the_add_lazyload_iframe' ) ) {
	function th90_core_the_add_lazyload_iframe( $iframe_html ) {

		echo th90_add_lazyload_iframe( $iframe_html );
	}
}

/**
 * -----------------------------------------------------------------------------
 * Disable Default Lazyload
 * -----------------------------------------------------------------------------
 */

if ( ! function_exists( 'th90_disable_lazy_load_default_images' ) ) {
	function th90_disable_lazy_load_default_images() {
		if ( th90_lazyload_check() ) {
			return false;
		}
		return true;
	}
}
add_filter('wp_lazy_loading_enabled', 'th90_disable_lazy_load_default_images');

if ( ! function_exists( 'th90_loading_eager' ) ) {
	function th90_loading_eager( $attr, $attachment = null ) {
		if ( th90_lazyload_check() ) {
			$attr['loading'] = 'eager';
		}

		if ( false !== strpos( $attr['class'], 'attachment-woocommerce_thumbnail' ) ) {
			$attr['loading'] = 'lazy';
		}
		return $attr;
	}
}
add_filter('wp_get_attachment_image_attributes', 'th90_loading_eager');
