<?php
namespace TH90Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( did_action( 'elementor/loaded' ) ) {
	require plugin_dir_path( __FILE__ ) . 'custom/custom.php';
}
/**
 * Main Elementor Class
 *
 */
final class TH90_Elementor {

	const MINIMUM_ELEMENTOR_VERSION = '3.5.0';

	const MINIMUM_PHP_VERSION = '7.3';

 	private static $_instance = null;

 	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

 	}

	public function __construct() {

		if ( $this->is_compatible() ) {
			add_action( 'elementor/init', [ $this, 'init' ] );
		}

	}

	public function is_compatible() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return false;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return false;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return false;
		}

		return true;
	}

	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'atlas-core' ),
			'<strong>' . esc_html__( 'Elementor Test Addon', 'atlas-core' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'atlas-core' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'atlas-core' ),
			'<strong>' . esc_html__( 'Elementor Test Addon', 'atlas-core' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'atlas-core' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'atlas-core' ),
			'<strong>' . esc_html__( 'Elementor Test Addon', 'atlas-core' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'atlas-core' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	public function init() {

		// Register editor scripts
		add_action( 'elementor/editor/after_enqueue_styles', [ $this, 'register_editor_styles' ] );

		// Register categories widget
		add_action( 'elementor/elements/categories_registered', [ $this, 'register_widget_categories' ] );

		// Register controls
		add_action( 'elementor/controls/register', [ $this, 'register_controls' ] );

		// Register widgets
		add_action( 'elementor/widgets/register', [ $this, 'register_widgets' ] );

	}

	public function register_editor_styles() {
		wp_register_style( 'th90-elementor-editor', plugins_url( 'assets/css/editor.css', __FILE__ ) );
		wp_enqueue_style( 'th90-elementor-editor' );
	}

	public function register_widget_categories( $elements_manager ) {
		$categories = [];
		$categories[ sanitize_key( wp_get_theme()->name ) . '-posts'] =
			[
				'title' => wp_get_theme()->name. ' - Posts',
				'icon'  => 'fa fa-plug'
			];
		$categories[ sanitize_key( wp_get_theme()->name ) . '-sliders'] =
			[
				'title' => wp_get_theme()->name. ' - Sliders & Ticker',
				'icon'  => 'fa fa-plug'
			];
		$categories[ sanitize_key( wp_get_theme()->name ) . '-groups'] =
			[
				'title' => wp_get_theme()->name. ' - Posts Groups',
				'icon'  => 'fa fa-plug'
			];
		$categories[ sanitize_key( wp_get_theme()->name ) . '-templates'] =
			[
				'title' => wp_get_theme()->name. ' - Templates',
				'icon'  => 'fa fa-plug'
			];

		$categories[ sanitize_key( wp_get_theme()->name ) . '-elements'] =
			[
				'title' => wp_get_theme()->name . ' - Elements & Widgets',
				'icon'  => 'fa fa-plug'
			];

		$old_categories = $elements_manager->get_categories();
		$categories = array_merge( $categories, $old_categories );

		$set_categories = function ( $categories ) {
			$this->categories = $categories;
		};

		$set_categories->call( $elements_manager, $categories );
	}

	public function register_controls( $controls_manager ) {

		require_once( __DIR__ . '/controls/choose.php' );

		$controls_manager->register( new \Control_Image_Choose() );

	}

	public function register_widgets( $widgets_manager ) {

		require_once( __DIR__ . '/custom/options.php' );
		require_once( __DIR__ . '/custom/controls.php' );

		$widgets = [
			/* Posts */
			'p-postsselective',
			'p-postsblog',
			'p-postssmall',
			'p-grid1',
			'p-grid2',

			/* Sliders */
			's-postsslider',
			's-postssmallslider',
			's-sliderthumb',
			's-sliderthumb2',
			's-ticker',

			/* Group */
			'g-postsgroup',
			'g-postsgroup2',
			'g-postsgroup3',
			'g-postsgroup4',
			'g-postsgroup5',
			'g-postsgroup6',

			/* Templates */
			't-block',
			't-sidebar',

			/* Elements & Widgets */
			'e-heading',
			'e-logo',
			'e-mainmenu',
			'e-triggersearch',
			'e-triggeroffcanvas',
			'e-triggerskin',
			'e-triggercart',
			'e-currentdate',
			'e-copyright',
			'e-button',
			'e-social',
			'e-divider',
			'w-tax',
			'w-taxcloud',
			//'w-about',
			'w-aboutus',
			'w-imagelightbox',
			'w-bannerbox',
			'w-socialcounters',
			'w-verticalmenu',
			'w-subscribebox',
			//'w-contact7',
			//'w-space',
			'w-customlist',
			'w-quotes',
			'w-comments',
			'w-searchform',
			'w-gallery',
        ];

		foreach ( $widgets as $widget_name ) {

			$file = __DIR__ . '/widgets/' . $widget_name . '.php';

			if( file_exists( $file ) ) {

				require_once $file;

				$class_name = str_replace( '-', ' ', $widget_name );
		        $class_name = ucwords( $class_name) ;
		        $class_name = str_replace( ' ', '_', $class_name );
		        $class_name = '\TH90_' . $class_name;

				if( class_exists( $class_name ) ) {
					$widgets_manager->register( new $class_name() );
				}
			}
		}
	}
}

// Instantiate
TH90_Elementor::instance();
