<?php

namespace WCFAddons\Widgets;

use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Typography;
use Elementor\Icons_Manager;
use Elementor\Repeater;
use Elementor\Group_Control_Background;
use Elementor\Utils;
use Elementor\Widget_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

/**
 * Pricing Table
 *
 * Elementor widget for pricing table.
 *
 * @since 1.0.0
 */
class Pricing_Table extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @return string Widget name.
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function get_name() {
		return 'wcf--pricing-table';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @return string Widget title.
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function get_title() {
		return esc_html__( 'WCF Pricing Table', 'wcf-addons' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function get_icon() {
		return 'wcf eicon-price-table';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @return array Widget categories.
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function get_categories() {
		return [ 'weal-coder-addon' ];
	}

	/**
	 * Requires css files.
	 *
	 * @return array
	 */
	public function get_style_depends() {
		return [
			'wcf--pricing-table',
		];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_layout',
			[
				'label' => esc_html__( 'Layout', 'wcf-addons' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'element_list',
			[
				'label'   => esc_html__( 'Style', 'wcf-addons' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '1',
				'options' => [
					'1' => esc_html__( 'One', 'wcf-addons' ),
//					'2' => esc_html__( 'Two', 'wcf-addons' ),
//					'3' => esc_html__( 'Three', 'wcf-addons' ),
//					'4' => esc_html__( 'Four', 'wcf-addons' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_layout',
			[
				'label'     => esc_html__( 'Layout', 'wcf-addons' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'element_list' => [ '1', '3' ],
				],
			]
		);

		$this->add_control(
			'layout_bg_color',
			[
				'label'     => esc_html__( 'Background Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wcf__priceTable.style-1' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'element_list' => '1',
				],
			]
		);

		$this->add_control(
			'layout_border_color',
			[
				'label'     => esc_html__( 'Border Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wcf__priceTable.style-1 .feature::after' => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .wcf__priceTable.style-3 .pt-name::after' => 'background-color: {{VALUE}}',
				],

			]
		);

		$this->end_controls_section();

		$this->price_heading_controls();

		$this->pricing_element_controls();

		$this->feature_list_controls();

		$this->button_controls();

		$this->ribbon_controls();
	}

	private function price_heading_controls() {
		$this->start_controls_section(
			'section_header',
			[
				'label' => esc_html__( 'Header', 'wcf-addons' ),
			]
		);

		$this->add_control(
			'heading_icon',
			[
				'label'            => esc_html__( 'Icon', 'wcf-addons' ),
				'type'             => Controls_Manager::ICONS,
				'fa4compatibility' => 'h_icon',
				'default'          => [
					'value'   => 'far fa-paper-plane',
					'library' => 'fa-solid',
				],
				'condition'        => [
					'element_list' => '1',
				],
			]
		);

		$this->add_control(
			'heading',
			[
				'label'   => esc_html__( 'Title', 'wcf-addons' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Enter your title', 'wcf-addons' ),
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'heading_tag',
			[
				'label'   => esc_html__( 'Title HTML Tag', 'wcf-addons' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
				],
				'default' => 'h3',
			]
		);

		$this->add_control(
			'number',
			[
				'label'     => esc_html__( 'Number', 'wcf-addons' ),
				'type'      => Controls_Manager::TEXT,
				'default'   => esc_html__( '01', 'wcf-addons' ),
				'separator' => 'before',
				'condition' => [
					'element_list' => '2',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_header_style',
			[
				'label'      => esc_html__( 'Header', 'wcf-addons' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			]
		);

		$this->add_control(
			'heading_heading_style',
			[
				'label' => esc_html__( 'Title', 'wcf-addons' ),
				'type'  => Controls_Manager::HEADING,
			]
		);

		$this->add_control(
			'heading_color',
			[
				'label'     => esc_html__( 'Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .name' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'heading_typography',
				'selector' => '{{WRAPPER}} .name',
			]
		);

		$this->add_control(
			'heading_number_style',
			[
				'label'     => esc_html__( 'Number', 'wcf-addons' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'element_list' => '2',
				],
			]
		);

		$this->add_control(
			'number_color',
			[
				'label'     => esc_html__( 'Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .number' => 'color: {{VALUE}}',
				],
				'condition' => [
					'element_list' => '2',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'number_typography',
				'selector'  => '{{WRAPPER}} .number',
				'condition' => [
					'element_list' => '2',
				],
			]
		);

		$this->add_control(
			'title_bg_color',
			[
				'label'     => esc_html__( 'Background Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wcf__priceTable.style-1 .pt-name' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'element_list' => '1',
				],
			]
		);

		$this->add_control(
			'heading_icon_style',
			[
				'label'     => esc_html__( 'Icon', 'wcf-addons' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'element_list' => '1',
				],
			]
		);

		$this->add_control(
			'hading_icon_color',
			[
				'label'     => esc_html__( 'Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pt-name .icon i'   => 'color: {{VALUE}}',
					'{{WRAPPER}} .pt-name .icon svg' => 'fill: {{VALUE}}',
				],
				'condition' => [
					'element_list' => '1',
				],
			]
		);

		$this->add_control(
			'hading_icon_size',
			[
				'label'     => esc_html__( 'Size', 'wcf-addons' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .pt-name .icon' => 'font-size: {{SIZE}}px;',
				],
				'condition' => [
					'element_list' => '1',
				],
			]
		);

		$this->end_controls_section();
	}

	private function pricing_element_controls() {
		$this->start_controls_section(
			'section_pricing',
			[
				'label' => esc_html__( 'Pricing', 'wcf-addons' ),
			]
		);

		$this->add_control(
			'currency_symbol',
			[
				'label'   => esc_html__( 'Currency Symbol', 'wcf-addons' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					''             => esc_html__( 'None', 'wcf-addons' ),
					'dollar'       => '&#36; ' . _x( 'Dollar', 'Currency', 'wcf-addons' ),
					'euro'         => '&#128; ' . _x( 'Euro', 'Currency', 'wcf-addons' ),
					'baht'         => '&#3647; ' . _x( 'Baht', 'Currency', 'wcf-addons' ),
					'franc'        => '&#8355; ' . _x( 'Franc', 'Currency', 'wcf-addons' ),
					'guilder'      => '&fnof; ' . _x( 'Guilder', 'Currency', 'wcf-addons' ),
					'krona'        => 'kr ' . _x( 'Krona', 'Currency', 'wcf-addons' ),
					'lira'         => '&#8356; ' . _x( 'Lira', 'Currency', 'wcf-addons' ),
					'peseta'       => '&#8359 ' . _x( 'Peseta', 'Currency', 'wcf-addons' ),
					'peso'         => '&#8369; ' . _x( 'Peso', 'Currency', 'wcf-addons' ),
					'pound'        => '&#163; ' . _x( 'Pound Sterling', 'Currency', 'wcf-addons' ),
					'real'         => 'R$ ' . _x( 'Real', 'Currency', 'wcf-addons' ),
					'ruble'        => '&#8381; ' . _x( 'Ruble', 'Currency', 'wcf-addons' ),
					'rupee'        => '&#8360; ' . _x( 'Rupee', 'Currency', 'wcf-addons' ),
					'indian_rupee' => '&#8377; ' . _x( 'Rupee (Indian)', 'Currency', 'wcf-addons' ),
					'shekel'       => '&#8362; ' . _x( 'Shekel', 'Currency', 'wcf-addons' ),
					'yen'          => '&#165; ' . _x( 'Yen/Yuan', 'Currency', 'wcf-addons' ),
					'won'          => '&#8361; ' . _x( 'Won', 'Currency', 'wcf-addons' ),
					'custom'       => esc_html__( 'Custom', 'wcf-addons' ),
				],
				'default' => 'dollar',
			]
		);

		$this->add_control(
			'currency_symbol_custom',
			[
				'label'     => esc_html__( 'Custom Symbol', 'wcf-addons' ),
				'type'      => Controls_Manager::TEXT,
				'condition' => [
					'currency_symbol' => 'custom',
				],
			]
		);

		$this->add_control(
			'price',
			[
				'label'   => esc_html__( 'Price', 'wcf-addons' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => '39.99',
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'sale',
			[
				'label'     => esc_html__( 'Sale', 'wcf-addons' ),
				'type'      => Controls_Manager::SWITCHER,
				'label_on'  => esc_html__( 'On', 'wcf-addons' ),
				'label_off' => esc_html__( 'Off', 'wcf-addons' ),
				'default'   => '',
			]
		);

		$this->add_control(
			'original_price',
			[
				'label'     => esc_html__( 'Original Price', 'wcf-addons' ),
				'type'      => Controls_Manager::NUMBER,
				'default'   => '59',
				'condition' => [
					'sale' => 'yes',
				],
				'dynamic'   => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'period',
			[
				'label'   => esc_html__( 'Period', 'wcf-addons' ),
				'type'    => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'default' => esc_html__( 'Monthly', 'wcf-addons' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_pricing_element_style',
			[
				'label'      => esc_html__( 'Pricing', 'wcf-addons' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			]
		);

		$this->add_control(
			'price_color',
			[
				'label'     => esc_html__( 'Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pt-sale-price, {{WRAPPER}} .pt-currency' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'price_typography',
				// Targeting also the .elementor-price-table class in order to get a higher specificity from the inline CSS.
				'selector' => '{{WRAPPER}} .pt-sale-price',
				'global'   => [
					'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
				],
			]
		);

		$this->add_control(
			'price_bg_color',
			[
				'label'     => esc_html__( 'Background Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wcf__priceTable.style-1 .pt-price-wrap' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'element_list' => '1',
				],
			]
		);

		$this->add_control(
			'heading_currency_style',
			[
				'label'     => esc_html__( 'Currency Symbol', 'wcf-addons' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'currency_symbol!' => '',
				],
			]
		);

		$this->add_control(
			'currency_size',
			[
				'label'     => esc_html__( 'Size', 'wcf-addons' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .pt-currency' => 'font-size: {{SIZE}}px;',
				],
				'condition' => [
					'currency_symbol!' => '',
				],
			]
		);

		$this->add_control(
			'heading_original_price_style',
			[
				'label'     => esc_html__( 'Original Price', 'wcf-addons' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'sale'            => 'yes',
					'original_price!' => '',
				],
			]
		);

		$this->add_control(
			'original_price_color',
			[
				'label'     => esc_html__( 'Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pt-org-price' => 'color: {{VALUE}}',
				],
				'condition' => [
					'sale'            => 'yes',
					'original_price!' => '',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'original_price_typography',
				'selector'  => '{{WRAPPER}} .pt-org-price',
				'condition' => [
					'sale'            => 'yes',
					'original_price!' => '',
				],
			]
		);

		$this->add_control(
			'heading_period_style',
			[
				'label'     => esc_html__( 'Period', 'wcf-addons' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'period!' => '',
				],
			]
		);

		$this->add_control(
			'period_color',
			[
				'label'     => esc_html__( 'Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pt-period' => 'color: {{VALUE}}',
				],
				'condition' => [
					'period!' => '',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'period_typography',
				'selector'  => '{{WRAPPER}} .pt-period',
				'condition' => [
					'period!' => '',
				],
			]
		);


		$this->end_controls_section();
	}

	private function ribbon_controls() {
		$this->start_controls_section(
			'section_ribbon',
			[
				'label' => esc_html__( 'Ribbon', 'wcf-addons' ),
			]
		);

		$this->add_control(
			'show_ribbon',
			[
				'label'     => esc_html__( 'Show', 'wcf-addons' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'ribbon_title',
			[
				'label'     => esc_html__( 'Title', 'wcf-addons' ),
				'type'      => Controls_Manager::TEXT,
				'default'   => esc_html__( 'Popular', 'wcf-addons' ),
				'condition' => [
					'show_ribbon' => 'yes',
				],
				'dynamic'   => [
					'active' => true,
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_ribbon_style',
			[
				'label'      => esc_html__( 'Ribbon', 'wcf-addons' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
				'condition'  => [
					'show_ribbon' => 'yes',
				],
			]
		);

		$this->add_control(
			'ribbon_text_color',
			[
				'label'     => esc_html__( 'Text Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wcf__priceTable .ribbon' => 'color: {{VALUE}}',
					'{{WRAPPER}} .wcf__priceTable .ribbon span' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'ribbon_bg_color',
			[
				'label'     => esc_html__( 'Background Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wcf__priceTable:not(.style-1) .ribbon'  => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .wcf__priceTable.style-1 .ribbon span'   => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .wcf__priceTable.style-1 .ribbon:before' => 'border-color: {{VALUE}}',
					'{{WRAPPER}} .wcf__priceTable.style-1 .ribbon:after'  => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();
	}

	private function button_controls() {
		$this->start_controls_section(
			'section_button',
			[
				'label' => esc_html__( 'Button', 'wcf-addons' ),
			]
		);

		$this->add_control(
			'button_text',
			[
				'label'   => esc_html__( 'Text', 'wcf-addons' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Click Here', 'wcf-addons' ),
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'link',
			[
				'label'       => esc_html__( 'Link', 'wcf-addons' ),
				'type'        => Controls_Manager::URL,
				'placeholder' => esc_html__( 'https://your-link.com', 'wcf-addons' ),
				'default'     => [
					'url' => '#',
				],
				'dynamic'     => [
					'active' => true,
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_button_style',
			[
				'label'      => esc_html__( 'Button', 'wcf-addons' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'typography',
				'selector' => '{{WRAPPER}} .pt-btn a',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'btn_border',
				'selector' => '{{WRAPPER}} .pt-btn a',
			]
		);

		$this->add_control(
			'btn_border_radius',
			[
				'label'      => esc_html__( 'Border Radius', 'wcf-addons' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .pt-btn a:not(.wcf-btn-oval)' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'  => [
					'element_list!' => '2',
				],
			]
		);

		$this->add_control(
			'btn_padding',
			[
				'label'      => esc_html__( 'Padding', 'wcf-addons' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .pt-btn a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs( 'tabs_button_style' );

		$this->start_controls_tab(
			'tab_button_normal',
			[
				'label' => esc_html__( 'Normal', 'wcf-addons' ),
			]
		);

		$this->add_control(
			'button_text_color',
			[
				'label'     => esc_html__( 'Text Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .pt-btn a' => 'fill: {{VALUE}}; color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'button_bg_color',
				'types'    => [ 'classic', 'gradient' ],
				'selector' => '{{WRAPPER}} .pt-btn a',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_button_hover',
			[
				'label' => esc_html__( 'Hover', 'wcf-addons' ),
			]
		);

		$this->add_control(
			'hover_color',
			[
				'label'     => esc_html__( 'Text Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .pt-btn a:hover, {{WRAPPER}} .pt-btn a:focus' => 'color: {{VALUE}};',
					'{{WRAPPER}} .pt-btn a:hover svg, {{WRAPPER}} .pt-btn a:focus svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'     => 'button_hover_bg_color',
				'types'    => [ 'classic', 'gradient' ],
				'selector' => '{{WRAPPER}} .pt-btn a span, {{WRAPPER}} .pt-btn a::after',
			]
		);

		$this->add_control(
			'button_hover_border_color',
			[
				'label'     => esc_html__( 'Border Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .pt-btn a:hover, {{WRAPPER}} .pt-btn a:focus' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();
	}

	private function feature_list_controls() {
		$this->start_controls_section(
			'section_features',
			[
				'label' => esc_html__( 'Features', 'wcf-addons' ),
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'item_text',
			[
				'label'   => esc_html__( 'Text', 'wcf-addons' ),
				'type'    => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'default' => esc_html__( 'List Item', 'wcf-addons' ),
			]
		);

		$default_icon = [
			'value'   => 'fas fa-check',
			'library' => 'fa-regular',
		];

		$repeater->add_control(
			'selected_item_icon',
			[
				'label'            => esc_html__( 'Icon', 'wcf-addons' ),
				'type'             => Controls_Manager::ICONS,
				'skin'             => 'inline',
				'label_block'      => false,
				'fa4compatibility' => 'item_icon',
				'default'          => $default_icon,
			]
		);

		$repeater->add_control(
			'item_icon_color',
			[
				'label'     => esc_html__( 'Icon Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}} i'   => 'color: {{VALUE}}',
					'{{WRAPPER}} {{CURRENT_ITEM}} svg' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'features_list',
			[
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => [
					[
						'item_text'          => esc_html__( 'Starter Pack Included', 'wcf-addons' ),
						'selected_item_icon' => $default_icon,
					],
					[
						'item_text'          => esc_html__( ' Budget Minimization', 'wcf-addons' ),
						'selected_item_icon' => $default_icon,
					],
					[
						'item_text'          => esc_html__( 'Venue Booking', 'wcf-addons' ),
						'selected_item_icon' => $default_icon,
					],
					[
						'item_text'          => esc_html__( 'Personal Trainer', 'wcf-addons' ),
						'selected_item_icon' => $default_icon,
					],
				],
				'title_field' => '{{{ item_text }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_features_list_style',
			[
				'label'      => esc_html__( 'Features', 'wcf-addons' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			]
		);

		$this->add_control(
			'features_list_color',
			[
				'label'     => esc_html__( 'Color', 'wcf-addons' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .feature li' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'features_list_typography',
				'selector' => '{{WRAPPER}} .feature li',
			]
		);

		$this->add_control(
			'features_icon_size',
			[
				'label'      => esc_html__( 'Icon Size', 'wcf-addons' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem', 'custom' ],
				'range'      => [
					'px' => [
						'min' => 1,
						'max' => 50,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .feature li i' => 'font-size: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'features_list_gap',
			[
				'label'      => esc_html__( 'Gap', 'wcf-addons' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem', 'custom' ],
				'range'      => [
					'px' => [
						'min' => 1,
						'max' => 50,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .feature li' => 'padding-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Retrieves the name of the highest priority template file that exists.
	 *
	 * @param string|array $template_names Template file(s) to search for, in order.
	 *
	 * @return string The template filename if one is located.
	 * @since 1.0.0
	 */
	protected function locate_template( $template_names ) {
		$located = '';
		foreach ( (array) $template_names as $template_name ) {
			if ( ! $template_name ) {
				continue;
			}

			if ( file_exists( dirname( __FILE__ ) . '/templates/' . $template_name ) ) {
				$located = dirname( __FILE__ ) . '/templates/' . $template_name;
				break;
			}
		}

		return $located;
	}

	/**
	 * Requires the template file with WordPress environment.
	 *
	 * @param string $_template_file Path to template file.
	 * @param bool $load_once Whether to require_once or require. Default true.
	 * @param array $args Optional. Additional arguments passed to the template.
	 *                               Default empty array.
	 *
	 * @since 1.0.0
	 */
	protected function load_template( $_template_file, $settings = array(), $load_once = false ) {
		if ( ! file_exists( $_template_file ) ) {
			return;
		}
		if ( $load_once ) {
			require_once $_template_file;
		} else {
			require $_template_file;
		}
	}

	/**
	 * Find template to render and includes it.
	 *
	 * @param string|array $slug template slug or args.
	 *
	 * @return void
	 */
	protected function render_template( $slug ) {
		$settings = $this->get_settings_for_display();

		$found_template = $this->locate_template( $slug );

		if ( ! $found_template ) {
			error_log(
				sprintf( '%s trying to load a template that is not exists.',
					$this->get_name(),
				)
			);
		}

		$this->load_template( $found_template, $settings );
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_render_attribute( 'wrapper', 'class', 'wcf__priceTable style-' . $settings['element_list'] );

		$template = 'template-' . $settings['element_list'] . '.php';
		$this->render_template( $template );
	}

	protected function render_ribbon() {
		if ( ! $this->get_settings( 'show_ribbon' ) ) {
			return;
		}

		if ( empty( $this->get_settings( 'ribbon_title' ) ) ) {
			return;
		}
		?>
		<div class="ribbon">
			<span><?php $this->print_unescaped_setting( 'ribbon_title' ); ?></span>
		</div>
		<?php
	}

	protected function render_button() {
		$ext_wrap = in_array( $this->get_settings( 'element_list' ), [ '2' ] );

		if ( $ext_wrap ) {
			$this->add_render_attribute( 'button', 'class',
				[ 'wcf-btn-oval', 'btn-item', 'btn-hover-bgchange' ]
			);
		}

		if ( '1' === $this->get_settings( 'element_list' ) ) {
			$this->add_render_attribute( 'button', 'class',
				[ 'wcf-btn-primary btn-hover-divide' ]
			);
		}

		if ( '3' === $this->get_settings( 'element_list' ) ) {
			$this->add_render_attribute( 'button', 'class',
				[ 'wcf-btn-border btn-hover-cross' ]
			);
		}

		if ( '4' === $this->get_settings( 'element_list' ) ) {
			$this->add_render_attribute( 'button', 'class',
				[ 'wcf-btn-primary btn-hover-cropping' ]
			);
		}

		if ( ! empty( $this->get_settings( 'link' )['url'] ) ) {
			$this->add_link_attributes( 'button', $this->get_settings( 'link' ) );
		} else {
			$this->add_render_attribute( 'button', 'role', 'button' );
		}
		?>
		<div class="pt-btn">
			<?php if ( $ext_wrap ) : ?>
			<div class="btn-wrapper">
			<?php endif; ?>
				<a <?php $this->print_render_attribute_string( 'button' ); ?>>
					<?php $this->print_unescaped_setting( 'button_text' ); ?>
				</a>
				<?php if ( $ext_wrap ) : ?>
			</div>
				<?php endif; ?>
		</div>
		<?php
	}

	protected function render_feature_list() {
		if ( empty( $this->get_settings( 'features_list' ) ) ) {
			return;
		}

		$settings          = $this->get_settings_for_display();
		$migration_allowed = Icons_Manager::is_migration_allowed();
		?>
		<div class="pt-feature">
			<ul class="feature">
				<?php
				foreach ( $settings['features_list'] as $index => $item ) :
					$migrated = isset( $item['__fa4_migrated']['selected_item_icon'] );
					// add old default
					if ( ! isset( $item['item_icon'] ) && ! $migration_allowed ) {
						$item['item_icon'] = 'fa fa-check';
					}
					$is_new = ! isset( $item['item_icon'] ) && $migration_allowed;
					?>
					<li class="elementor-repeater-item-<?php echo esc_attr( $item['_id'] ); ?>">
						<?php if ( ! empty( $item['item_icon'] ) || ! empty( $item['selected_item_icon'] ) ) :
							if ( $is_new || $migrated ) :
								Icons_Manager::render_icon( $item['selected_item_icon'], [ 'aria-hidden' => 'true' ] );
							else : ?>
								<i class="<?php echo esc_attr( $item['item_icon'] ); ?>" aria-hidden="true"></i>
							<?php
							endif;
						endif; ?>
						<?php if ( ! empty( $item['item_text'] ) ) : ?>
							<?php $this->print_unescaped_setting( 'item_text', 'features_list', $index ); ?>
						<?php
						else :
							echo '&nbsp;';
						endif;
						?>
					</li>
				<?php endforeach; ?>
			</ul>
		</div>
		<?php
	}

	private function render_currency_symbol() {
		$settings = $this->get_settings_for_display();
		$symbol   = '';
		if ( ! empty( $settings['currency_symbol'] ) ) {
			if ( 'custom' !== $settings['currency_symbol'] ) {
				$symbol = $this->get_currency_symbol( $settings['currency_symbol'] );
			} else {
				$symbol = $settings['currency_symbol_custom'];
			}
		}

		if ( empty( $symbol ) ) {
			return;
		}
		echo esc_html( $symbol );
	}

	private function get_currency_symbol( $symbol_name ) {
		$symbols = [
			'dollar'       => '&#36;',
			'euro'         => '&#128;',
			'franc'        => '&#8355;',
			'pound'        => '&#163;',
			'ruble'        => '&#8381;',
			'shekel'       => '&#8362;',
			'baht'         => '&#3647;',
			'yen'          => '&#165;',
			'won'          => '&#8361;',
			'guilder'      => '&fnof;',
			'peso'         => '&#8369;',
			'peseta'       => '&#8359',
			'lira'         => '&#8356;',
			'rupee'        => '&#8360;',
			'indian_rupee' => '&#8377;',
			'real'         => 'R$',
			'krona'        => 'kr',
		];

		return isset( $symbols[ $symbol_name ] ) ? $symbols[ $symbol_name ] : '';
	}

	protected function render_price_heading() {
		if ( empty( $this->get_settings( 'heading' ) ) ) {
			return;
		}
		$heading_tag = Utils::validate_html_tag( $this->get_settings( 'heading_tag' ) );
		?>
		<<?php Utils::print_validated_html_tag( $heading_tag ); ?> class="name">
		<?php $this->print_unescaped_setting( 'heading' ); ?>
		</<?php Utils::print_validated_html_tag( $heading_tag ); ?>>
		<?php
	}

	protected function render_original_price() {
		if ( ! $this->get_settings( 'sale' ) ) {
			return;
		}
		?>
		<span class="pt-org-price">
		<?php
		$this->render_currency_symbol();
		$this->print_unescaped_setting( 'original_price' );
		?>
		</span>
		<?php
	}

	protected function render_header_icon() {
		$settings = $this->get_settings_for_display();

		$migrated = isset( $settings['__fa4_migrated']['heading_icon'] );
		$is_new   = empty( $settings['h_icon'] ) && Icons_Manager::is_migration_allowed();
		?>
		<div class="icon">
			<?php
			if ( $is_new || $migrated ) :
				Icons_Manager::render_icon( $settings['heading_icon'], [ 'aria-hidden' => 'true' ] );
			else : ?>
				<i class="<?php echo esc_attr( $settings['h_icon'] ); ?>" aria-hidden="true"></i>
			<?php endif; ?>
		</div>
		<?php
	}
}
