<?php

namespace WCFAddons;

use Elementor\Plugin as ElementorPlugin;

/**
 * Class Plugin
 *
 * Main Plugin class
 *
 * @since 1.2.0
 */
class Plugin {

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 * @access private
	 * @static
	 *
	 * @var Plugin The single instance of the class.
	 */
	private static $instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @return Plugin An instance of the class.
	 * @since 1.2.0
	 * @access public
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Widget_scripts
	 *
	 * Load required plugin core files.
	 *
	 * @since 1.2.0
	 * @access public
	 */
	public function widget_scripts() {
		$scripts = [
			'gsap'            => [
				'handler' => 'gsap',
				'src'     => 'gsap.min.js',
				'dep'     => [],
				'version' => false,
				'arg'     => true,
			],
			'scrollSmoother'  => [
				'handler' => 'scrollSmoother',
				'src'     => 'ScrollSmoother.min.js',
				'dep'     => [ 'gsap' ],
				'version' => false,
				'arg'     => true,
			],
			'scrollTrigger'   => [
				'handler' => 'scrollTrigger',
				'src'     => 'ScrollTrigger.min.js',
				'dep'     => [ 'gsap' ],
				'version' => false,
				'arg'     => true,
			],
			'scrollTo'   => [
				'handler' => 'scrollTo',
				'src'     => 'ScrollToPlugin.min.js',
				'dep'     => [ 'gsap' ],
				'version' => false,
				'arg'     => true,
			],
			'splitText'       => [
				'handler' => 'split-text',
				'src'     => 'SplitText.min.js',
				'dep'     => [ 'gsap' ],
				'version' => false,
				'arg'     => true,
			],
			'magnific-popup'  => [
				'handler' => 'magnific-popup',
				'src'     => 'jquery.magnific-popup.min.js',
				'dep'     => [],
				'version' => false,
				'arg'     => true,
			],
			'wcf-addons-core' => [
				'handler' => 'wcf--addons',
				'src'     => 'wcf-addons.js',
				'dep'     => [ 'jquery', 'gsap', 'magnific-popup', 'scrollSmoother', 'scrollTo', 'scrollTrigger', 'split-text' ],
				'version' => false,
				'arg'     => true,
			],
		];

		foreach ( $scripts as $key => $script ) {
			wp_register_script( $script['handler'], plugins_url( '/assets/js/' . $script['src'], __FILE__ ), $script['dep'], $script['version'], $script['arg'] );
		}

		$data = apply_filters( 'wcf-addons/js/data', [
			'ajaxUrl'  => admin_url( 'admin-ajax.php' ),
			'_wpnonce' => wp_create_nonce( 'wcf-addons-frontend' ),
			'post_id'  => get_the_ID(),
			'i18n'     => [
				'okay'    => esc_html__( 'Okay', 'wcf-addons' ),
				'cancel'  => esc_html__( 'Cancel', 'wcf-addons' ),
				'submit'  => esc_html__( 'Submit', 'wcf-addons' ),
				'success' => esc_html__( 'Success', 'wcf-addons' ),
				'warning' => esc_html__( 'Warning', 'wcf-addons' ),
			],
		] );

		wp_localize_script( 'wcf--addons', 'WCF_ADDONS_JS', $data );

		wp_enqueue_script( 'wcf--addons' );

		//widget scripts
		foreach ( self::get_widget_scripts() as $key => $script ) {
			wp_register_script( $script['handler'], plugins_url( '/assets/js/' . $script['src'], __FILE__ ), $script['dep'], $script['version'], $script['arg'] );
		}
	}

	/**
	 * Function widget_styles
	 *
	 * Load required plugin core files.
	 *
	 * @since 1.2.0
	 * @access public
	 */
	public static function widget_styles() {
		$styles = [
			'magnific-popup'   => [
				'handler' => 'magnific-popup',
				'src'     => 'magnific-popup.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'wcf-addons-core' => [
				'handler' => 'wcf--addons',
				'src'     => 'wcf-addons.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
		];

		foreach ( $styles as $key => $style ) {
			wp_register_style( $style['handler'], plugins_url( '/assets/css/' . $style['src'], __FILE__ ), $style['dep'], $style['version'], $style['media'] );
		}

		wp_enqueue_style( 'wcf--addons' );

		//widget style
		foreach ( self::get_widget_style() as $key => $style ) {
			wp_register_style( $style['handler'], plugins_url( '/assets/css/' . $style['src'], __FILE__ ), $style['dep'], $style['version'], $style['media'] );
		}
	}

	/**
	 * Editor scripts
	 *
	 * Enqueue plugin javascripts integrations for Elementor editor.
	 *
	 * @since 1.2.1
	 * @access public
	 */
	public function editor_scripts() {
		wp_enqueue_script( 'wcf-editor', plugins_url( '/assets/js/editor.js', __FILE__ ), [
			'elementor-editor',
		], WCF_ADDONS_VERSION, true );

		$data = apply_filters( 'wcf-addons-editor/js/data', [
			'ajaxUrl'  => admin_url( 'admin-ajax.php' ),
			'_wpnonce' => wp_create_nonce( 'wcf-addons-editor' ),
		] );

		wp_localize_script( 'wcf-editor', 'WCF_Addons_Editor', $data );
	}

	/**
	 * Editor style
	 *
	 * Enqueue plugin css integrations for Elementor editor.
	 *
	 * @since 1.2.1
	 * @access public
	 */
	public function editor_styles() {
		wp_enqueue_style( 'wcf--editor', plugins_url( '/assets/css/editor.css', __FILE__ ), [], WCF_ADDONS_VERSION, 'all' );
	}

	/**
	 * Force load editor script as a module
	 *
	 * @param string $tag
	 * @param string $handle
	 *
	 * @return string
	 * @since 1.2.1
	 */
	public function editor_scripts_as_a_module( $tag, $handle ) {
		if ( 'elementor-hello-world-editor' === $handle ) {
			$tag = str_replace( '<script', '<script type="module"', $tag );
		}

		return $tag;
	}

	/**
	 * Function widget_scripts
	 *
	 * Load required plugin core files.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public static function get_widget_scripts() {
		return [
			'typed'            => [
				'handler' => 'typed',
				'src'     => 'typed.js',
				'dep'     => [],
				'version' => false,
				'arg'     => true,
			],
			'mixitup'          => [
				'handler' => 'mixitup',
				'src'     => 'mixitup.min.js',
				'dep'     => [],
				'version' => false,
				'arg'     => true,
			],
			'ProgressBar'      => [
				'handler' => 'progressbar',
				'src'     => 'progressbar.min.js',
				'dep'     => [],
				'version' => false,
				'arg'     => true,
			],
			'slider'           => [
				'handler' => 'wcf--slider',
				'src'     => 'widgets/slider.js',
				'dep'     => [],
				'version' => false,
				'arg'     => true,
			],
			'typewriter'       => [
				'handler' => 'wcf--typewriter',
				'src'     => 'widgets/typewriter.js',
				'dep'     => [ 'typed', 'jquery' ],
				'version' => false,
				'arg'     => true,
			],
			'text-hover-image' => [
				'handler' => 'wcf--text-hover-image',
				'src'     => 'widgets/text-hover-image.js',
				'dep'     => [ 'jquery' ],
				'version' => false,
				'arg'     => true,
			],
			'counter'          => [
				'handler' => 'wcf--counter',
				'src'     => 'widgets/counter.js',
				'dep'     => [ 'jquery-numerator' ],
				'version' => false,
				'arg'     => true,
			],
			'portfolio'        => [
				'handler' => 'wcf--portfolio',
				'src'     => 'widgets/portfolio.js',
				'dep'     => [ 'mixitup' ],
				'version' => false,
				'arg'     => true,
			],
			'progressbar'      => [
				'handler' => 'wcf--progressbar',
				'src'     => 'widgets/progressbar.js',
				'dep'     => [ 'progressbar' ],
				'version' => false,
				'arg'     => true,
			],
			'before-after'     => [
				'handler' => 'beforeAfter',
				'src'     => 'beforeafter.jquery-1.0.0.min.js',
				'dep'     => [ 'jquery' ],
				'version' => false,
				'arg'     => true,
			],
			'image-compare'    => [
				'handler' => 'wcf--image-compare',
				'src'     => 'widgets/image-compare.js',
				'dep'     => [ 'beforeAfter' ],
				'version' => false,
				'arg'     => true,
			],
			'mailchimp'        => [
				'handler' => 'wcf--mailchimp',
				'src'     => 'widgets/mailchimp.js',
				'dep'     => [],
				'version' => false,
				'arg'     => true,
			],
			'tabs'             => [
				'handler' => 'wcf--tabs',
				'src'     => 'widgets/tabs.js',
				'dep'     => [ 'jquery' ],
				'version' => false,
				'arg'     => true,
			],
			'chroma' => [
				'handler' => 'chroma',
				'src'     => 'chroma.min.js',
				'dep'     => [ 'jquery', 'gsap' ],
				'version' => false,
				'arg'     => true,
			],
			'animated-heading' => [
				'handler' => 'wcf--animated-heading',
				'src'     => 'widgets/animated-heading.js',
				'dep'     => [ 'jquery', 'gsap', 'chroma' ],
				'version' => false,
				'arg'     => true,
			],
		];
	}

	/**
	 * Function widget_style
	 *
	 * Load required plugin core files.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public static function get_widget_style() {
		return [
			'icon-box'         => [
				'handler' => 'wcf--icon-box',
				'src'     => 'widgets/icon-box.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'testimonial'      => [
				'handler' => 'wcf--testimonial',
				'src'     => 'widgets/testimonial.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'testimonial2'     => [
				'handler' => 'wcf--testimonial2',
				'src'     => 'widgets/testimonial2.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'testimonial3'     => [
				'handler' => 'wcf--testimonial3',
				'src'     => 'widgets/testimonial3.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'portfolio'        => [
				'handler' => 'wcf--portfolio',
				'src'     => 'widgets/portfolio.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'posts'            => [
				'handler' => 'wcf--posts',
				'src'     => 'widgets/posts.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'button'           => [
				'handler' => 'wcf--button',
				'src'     => 'widgets/button.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'pricing-table'    => [
				'handler' => 'wcf--pricing-table',
				'src'     => 'widgets/pricing-table.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'progressbar'      => [
				'handler' => 'wcf--progressbar',
				'src'     => 'widgets/progressbar.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'counter'          => [
				'handler' => 'wcf--counter',
				'src'     => 'widgets/counter.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'image-compare'    => [
				'handler' => 'wcf--image-compare',
				'src'     => 'widgets/image-compare.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'brand-slider'     => [
				'handler' => 'wcf--brand-slider',
				'src'     => 'widgets/brand-slider.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'text-hover-image' => [
				'handler' => 'wcf--text-hover-image',
				'src'     => 'widgets/text-hover-image.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'one-page-nav'     => [
				'handler' => 'wcf--one-page-nav',
				'src'     => 'widgets/one-page-nav.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'video-box'        => [
				'handler' => 'wcf--video-box',
				'src'     => 'widgets/video-box.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'social-icons'     => [
				'handler' => 'wcf--social-icons',
				'src'     => 'widgets/social-icons.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'image-gallery'    => [
				'handler' => 'wcf--image-gallery',
				'src'     => 'widgets/image-gallery.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'team'             => [
				'handler' => 'wcf--team',
				'src'     => 'widgets/team.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'image-box'        => [
				'handler' => 'wcf--image-box',
				'src'     => 'widgets/image-box.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'timeline'         => [
				'handler' => 'wcf--timeline',
				'src'     => 'widgets/timeline.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'mailchimp'        => [
				'handler' => 'wcf--mailchimp',
				'src'     => 'widgets/mailchimp.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'event-slider'     => [
				'handler' => 'wcf--event-slider',
				'src'     => 'widgets/event-slider.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'services-tab'     => [
				'handler' => 'wcf--services-tab',
				'src'     => 'widgets/services-tab.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'video-box-slider' => [
				'handler' => 'wcf--video-box-slider',
				'src'     => 'widgets/video-box-slider.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'countdown'        => [
				'handler' => 'wcf--countdown',
				'src'     => 'widgets/countdown.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
			'video-mask'        => [
				'handler' => 'wcf--video-mask',
				'src'     => 'widgets/video-mask.css',
				'dep'     => [],
				'version' => false,
				'media'   => 'all',
			],
		];
	}

	/**
	 * Register Widgets
	 *
	 * Register new Elementor widgets.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function register_widgets() {
		foreach ( self::get_widgets() as $slug => $data ) {

			// If upcoming don't register.
			if ( $data['is_upcoming'] ) {
				continue;
			}

			if ( $data['is_active'] ) {
				if ( is_dir( __DIR__ . '/widgets/' . $slug ) ) {
					require_once( __DIR__ . '/widgets/' . $slug . '/' . $slug . '.php' );
				} else {
					require_once( __DIR__ . '/widgets/' . $slug . '.php' );
				}


				$class = explode( '-', $slug );
				$class = array_map( 'ucfirst', $class );
				$class = implode( '_', $class );
				$class = 'WCFAddons\\Widgets\\' . $class;
				ElementorPlugin::instance()->widgets_manager->register( new $class() );
			}
		}
	}

	/**
	 * Get Widgets List.
	 *
	 * @return array
	 */
	public static function get_widgets() {
		return apply_filters( 'wcf_widgets', [
			'image-box'         => [
				'label'       => __( 'Image Box', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'image-box-slider'  => [
				'label'       => __( 'Image Box Slider', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'typewriter'        => [
				'label'       => __( 'Typewriter', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'animated-title'    => [
				'label'       => __( 'Animated Title', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'animated-text'     => [
				'label'       => __( 'Animated Text', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'social-icons'      => [
				'label'       => __( 'Social Icons', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'image'             => [
				'label'       => __( 'Image', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'image-gallery'     => [
				'label'       => __( 'mage Gallery', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'text-hover-image'  => [
				'label'       => __( 'Text Hover Image', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'brand-slider'      => [
				'label'       => __( 'Brand Slider', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'counter'           => [
				'label'       => __( 'Counter', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'icon-box'          => [
				'label'       => __( 'Icon Box', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'testimonial'       => [
				'label'       => __( 'Testimonial', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'testimonial2'      => [
				'label'       => __( 'Testimonial 2', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'testimonial3'      => [
				'label'       => __( 'Testimonial 3', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'portfolio'         => [
				'label'       => __( 'Portfolio', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'posts'             => [
				'label'       => __( 'Posts', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'button'            => [
				'label'       => __( 'Button', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'pricing-table'     => [
				'label'       => __( 'Pricing Table', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'image-compare'     => [
				'label'       => __( 'Image Compare', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'progressbar'       => [
				'label'       => __( 'Progressbar', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'video-popup'       => [
				'label'       => __( 'Video Popup', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'team'              => [
				'label'       => __( 'Team', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'one-page-nav'      => [
				'label'       => __( 'One Page Nav', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'timeline'          => [
				'label'       => __( 'Timeline', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'video-box'         => [
				'label'       => __( 'Video Box', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'contact-form-7'    => [
				'label'       => __( 'Contact Form 7', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'mailchimp'         => [
				'label'       => __( 'Mailchimp', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'tabs'              => [
				'label'       => __( 'Tabs', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'services-tab'      => [
				'label'       => __( 'Services Tabs', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'floating-elements' => [
				'label'       => __( 'Floating Elements', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'event-slider'      => [
				'label'       => __( 'Event Slider', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'video-box-slider'  => [
				'label'       => __( 'Video Box Slider', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'content-slider'    => [
				'label'       => __( 'Content Slider', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'countdown'         => [
				'label'       => __( 'Countdown', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'video-mask'         => [
				'label'       => __( 'Video Mask', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
			'animated-heading'         => [
				'label'       => __( 'Animated Heading', 'wcf-addons' ),
				'is_active'   => true,
				'is_upcoming' => false,
				'demo_url'    => '',
				'doc_url'     => '',
				'youtube_url' => '',
			],
		] );
	}

	/**
	 * Widget Category
	 *
	 * @param $elements_manager
	 */
	public function widget_categories( $elements_manager ) {
		$categories = [];

		$categories['weal-coder-addon'] = [
			'title' => esc_html__( 'WCF', 'wcf-addons' ),
			'icon'  => 'fa fa-plug',
		];

		$old_categories = $elements_manager->get_categories();
		$categories     = array_merge( $categories, $old_categories );

		$set_categories = function ( $categories ) {
			$this->categories = $categories;
		};

		$set_categories->call( $elements_manager, $categories );
	}

	/**
	 * Include Plugin files
	 *
	 * @access private
	 */
	private function include_files() {
		require_once WCF_ADDONS_PATH . 'inc/helper.php';
		require_once WCF_ADDONS_PATH . 'inc/hook.php';
		require_once WCF_ADDONS_PATH . 'inc/ajax-handler.php';
		include_once WCF_ADDONS_PATH . 'inc/wcf-animation-effects.php';
		include_once WCF_ADDONS_PATH . 'inc/trait-wcf-post-query.php';
		include_once WCF_ADDONS_PATH . 'inc/trait-wcf-button.php';
		include_once WCF_ADDONS_PATH . 'inc/trait-wcf-slider.php';
		include_once WCF_ADDONS_PATH . 'inc/class-wcf-custom-css.php';
	}

	/**
	 *  Plugin class constructor
	 *
	 * Register plugin action hooks and filters
	 *
	 * @since 1.2.0
	 * @access public
	 */
	public function __construct() {
		add_action( 'elementor/elements/categories_registered', [ $this, 'widget_categories' ] );

		// Register widget scripts
		add_action( 'wp_enqueue_scripts', [ $this, 'widget_scripts' ] );

		// Register widget style
		add_action( 'wp_enqueue_scripts', [ $this, 'widget_styles' ] );

		// Register widgets
		add_action( 'elementor/widgets/register', [ $this, 'register_widgets' ] );

		// Register editor scripts
		add_action( 'elementor/editor/after_enqueue_scripts', [ $this, 'editor_scripts' ] );

		// Register editor style
		add_action( 'elementor/editor/after_enqueue_styles', [ $this, 'editor_styles' ] );

		$this->include_files();
	}
}

// Instantiate Plugin Class
Plugin::instance();
